#!/bin/bash

set -euo pipefail

IPTABLES_V4="/sbin/iptables"
IPTABLES_V6="/sbin/ip6tables"
RULES_V4="/etc/iptables/rules.v4"
RULES_V6="/etc/iptables/rules.v6"

# -----------------------------
# Utility functions
# -----------------------------

require_root() {
    if [ "$(id -u)" -ne 0 ]; then
        echo "Error: This script must be run as root."
        exit 1
    fi
}

save_rules() {
    echo "Saving iptables rules..."
    iptables-save > "$RULES_V4"
    if command -v ip6tables-save >/dev/null 2>&1; then
        ip6tables-save > "$RULES_V6"
    fi
}

rule_exists() {
    local ip="$1"
    local proto="$2"
    local port="$3"

    if [ -z "$port" ]; then
        $IPTABLES_V4 -C INPUT -s "$ip" -j DROP 2>/dev/null
    else
        $IPTABLES_V4 -C INPUT -s "$ip" -p "$proto" --dport "$port" -j DROP 2>/dev/null
    fi
}

# -----------------------------
# Add ban
# -----------------------------

add_ban() {
    local ip="$1"
    local ports="${2:-}"

    if [ -z "$ip" ]; then
        echo "Error: No IP address specified."
        exit 1
    fi

    if [ -z "$ports" ]; then
        if rule_exists "$ip" "" ""; then
            echo "Rule already exists: $ip (all ports)"
        else
            echo "Banning $ip on all ports"
            $IPTABLES_V4 -A INPUT -s "$ip" -j DROP
        fi
    else
        IFS=',' read -ra PORT_ARRAY <<< "$ports"
        for port in "${PORT_ARRAY[@]}"; do
            for proto in tcp udp; do
                if rule_exists "$ip" "$proto" "$port"; then
                    echo "Rule already exists: $ip $proto/$port"
                else
                    echo "Banning $ip on $proto port $port"
                    $IPTABLES_V4 -A INPUT -s "$ip" -p "$proto" --dport "$port" -j DROP
                fi
            done
        done
    fi

    save_rules
}

# -----------------------------
# Remove ban
# -----------------------------

remove_ban() {
    local ip="$1"
    local ports="${2:-}"

    if [ "$ip" = "all" ]; then
        echo "Flushing INPUT chain (IPv4 only)"
        $IPTABLES_V4 -F INPUT
        save_rules
        return
    fi

    if [ -z "$ip" ]; then
        echo "Error: No IP address specified."
        exit 1
    fi

    if [ -z "$ports" ]; then
        echo "Removing all DROP rules for $ip"
        while $IPTABLES_V4 -D INPUT -s "$ip" -j DROP 2>/dev/null; do :; done
    else
        IFS=',' read -ra PORT_ARRAY <<< "$ports"
        for port in "${PORT_ARRAY[@]}"; do
            for proto in tcp udp; do
                while $IPTABLES_V4 -D INPUT -s "$ip" -p "$proto" --dport "$port" -j DROP 2>/dev/null; do :; done
            done
        done
    fi

    save_rules
}

# -----------------------------
# List bans
# -----------------------------

list_bans() {
    echo "Active DROP rules (INPUT chain):"
    echo

    local found=false
    declare -A bans

    while read -r line; do
        ip=$(echo "$line" | awk '{print $5}')
        port=$(echo "$line" | grep -oP 'dpt:\K\d+')

        if [ -n "$ip" ]; then
            found=true
            if [ -n "$port" ]; then
                bans["$ip"]+="$port "
            else
                bans["$ip"]="all ports"
            fi
        fi
    done < <($IPTABLES_V4 -L INPUT -n | grep DROP)

    if ! $found; then
        echo "No bans present."
        return
    fi

    for ip in "${!bans[@]}"; do
        echo "$ip: ${bans[$ip]}"
    done
}

# -----------------------------
# Main
# -----------------------------

require_root

if [ "$#" -lt 1 ]; then
    echo "Usage:"
    echo "  $0 add <ip> [port1,port2]"
    echo "  $0 remove <ip|all> [port1,port2]"
    echo "  $0 list"
    exit 1
fi

ACTION="$1"
IP="${2:-}"
PORTS="${3:-}"

case "$ACTION" in
    add)
        add_ban "$IP" "$PORTS"
        ;;
    remove)
        remove_ban "$IP" "$PORTS"
        ;;
    list)
        list_bans
        ;;
    *)
        echo "Invalid action: $ACTION"
        exit 1
        ;;
esac